//#include <msp430.h>
#include <msp430g2553.h>
#include <stdint.h>

#include "attack.h"
#include "sustain.h"
#include "envelope.h"
#include "scales.h"
#include "tune.h"

#define SAMPLERATE 7812
#define ENVELOPEDURATION 1500 // miliseconds
#define OSCILLATOR_COUNT 6 // number of simultaneous notes, 256 max or 8 bit accumulation will overflow 16 bits
#define SIMULTANEOUS_NOTES 6 // generated samples are divided by this value, lots of clipping will happen if too low, volume will be low if too high
#define CLIP 127

#define BPM 190 // beats per minute
#define QUARTER 120 // note resolution in midi file
#define TICKS_LIMIT 20 // SAMPLERATE * 60ul / (BPM * QUARTER), ticks in a time unit of the tune
#define sustainfreq (SAMPLERATE / sizeof_wt_sustain)

// some data use fixed point values, the 4 lower bits are used as the value after the decimal point
// round is achieved by dividing by 16 (or doing a right shift of 4 bits on unsigned values)

volatile char nboverflow=0;

// configure timer to generate pwm on pin 0
void setup() {
	/* Disable the watchdog. The watchdog is on by default on an MSP430, if we do not disable it it will reset the CPU after
	* 32k cycles in the standard configuration.	*/
	WDTCTL = WDTPW + WDTHOLD;
	// set P1.6 as output (and set other unused pins as outputs)
	P1DIR = 255;
	P2DIR = 255;
	P3DIR = 255;

	TACCTL1	= OUTMOD_7;            // TACCR1 reset/set
	TACTL	= TASSEL_2 + MC_1;     // SMCLK, up mode
	TACCTL0 = CCIE;					// compare interrupt enabled
	TACCR0	= 255;        			// PWM Period
	TACCR1	= 128;            		// PWM Duty Cycle
	P1SEL |= BIT6;				// P1.6 = TA1 output
    _BIS_SR(GIE);				// enable interrupts
    DCOCTL = CALDCO_16MHZ;		// SET clock to 16MHz
    BCSCTL1 = CALBC1_16MHZ;
}

//  ======== TIMERA Interrupt Service Routine ========
#pragma vector=TIMER0_A0_VECTOR
__interrupt void TA0_ISR(void)
{
	// is called every pwm period, 1/62500 s (16 microseconds)
	nboverflow++;
}

int main() {
	setup();

    uint16_t increments[ OSCILLATOR_COUNT ]; // 4 lower bits are fixed point value
    uint32_t phase_accu[ OSCILLATOR_COUNT ]; // same
    uint16_t envelope_positions[ OSCILLATOR_COUNT ]; // same
    uint8_t sustaining[ OSCILLATOR_COUNT ];  // true if the attack is finished and the channel is now using the sustain wavetable
    uint8_t playing [ OSCILLATOR_COUNT ];    // true if the channel is playing, playing stops when the sound has faded because of the envelope

    uint8_t next_osc = 0; // the next free (or not) channel
    uint16_t ticks = 0; // each tick we compute a sample
    uint16_t time = 120;  // tune time
    uint16_t event_index = 0; // index in the tune
    uint32_t totalticks =0;

    const uint16_t event_count = sizeof(tune_pitch);

    const uint8_t envelopeinc=2; // = sizeof_envelope_table*256ul*1000ul /ENVELOPEDURATION/SAMPLERATE;

    // initialization, set everything to 0
    uint8_t osc;
    for ( osc = 0; osc < OSCILLATOR_COUNT; ++osc ) {
        increments[ osc ] = 0;
        phase_accu[ osc ] = 0;
        envelope_positions[ osc ] = 0;
	sustaining[ osc ] = 0;
	playing [ osc ] = 0;
    }

    while ( 1 ) {
        while ( time >= tune_time[event_index] ) { // time reached the next note(s), let's assign an oscillator to it(them)
			// base sample frequency = SAMPLERATE / sizeof_wt_sustain
			// increment = note frequency / base frequency
            uint8_t pitch = tune_pitch[event_index]; // play one octave higher
            increments[ next_osc ] = (scale_table[pitch] << 4) / sustainfreq;
            phase_accu[ next_osc ] = 0;
            envelope_positions[ next_osc ] = 0;
	    sustaining [ next_osc ] = 0;
	    playing [ next_osc ] = 1;
            ++next_osc;
            if ( next_osc >= OSCILLATOR_COUNT ) {
                next_osc = 0;
            }
            ++event_index;
	  // if end of the song, restart
            if ( event_index >= event_count ) {
                ticks = 0;
                time = 0;
                event_index = 0;
            }
        }
        ++ticks;
        totalticks++;
        if ( ticks >= TICKS_LIMIT ) {
	  ticks=0;
            time += 1;
        }

        int16_t value = 0;
        for ( osc = 0; osc < OSCILLATOR_COUNT; ++osc ) {
			if (playing [ osc ] ) {

				phase_accu[ osc ] += increments[ osc ]; // advance phase
                                uint8_t envelope;
                                int8_t sustain;
				if (sustaining [ osc]) { // if sustaining
					while ( (phase_accu[ osc ] >> 4 )>= sizeof_wt_sustain ) { // check if beyond table
						phase_accu[ osc ] -= sizeof_wt_sustainx256;
					}
                                        uint16_t envpos = envelope_positions[osc] >>4;
					if ( (envpos) >= sizeof_envelope_table - 1 ) {
						// sound has decayed, no need to play it anymore.
						playing [ osc ] = 0;
					} else {
                                                envelope = envelope_table [envpos];
                                                sustain = wt_sustain [phase_accu[osc]>>4];
                                                int16_t temp;
                                                if (sustain < 0) {
                                                  temp = -sustain;
                                                  temp *=envelope;
                                                  value -= temp >> 8;
                                                } else {
                                                  temp = sustain;
                                                  temp *= envelope;
						  value += temp >> 8; // apply envelope only if sustaining
                                                }
						envelope_positions [ osc ] += envelopeinc;
					}
				} else {
					if ( (phase_accu[ osc ] >>4) >= sizeof_wt ) { // check if beyond table
						sustaining [ osc ] = 1;
						phase_accu[ osc ] -= sizeof_wtx256;
						// check if beyond sustain table too
						while ( (phase_accu[ osc ] >> 4) >= sizeof_wt_sustain ) {
							phase_accu[ osc ] -= sizeof_wt_sustainx256;
						}
						sustain = wt_sustain [phase_accu[osc]>>4];
						value += sustain;
					} else {
						sustain = wt_attack [phase_accu[osc]>>4];
						value += sustain;
					}
				}
			}
        }
        value /= SIMULTANEOUS_NOTES;
        if ( value > CLIP ) {
            value = CLIP;
        } else if ( value < -CLIP ) {
            value = -CLIP;
        }
        value += 128;
        while (nboverflow <8);
        TACCR1 = value;
        nboverflow=0;
    }
}
