#define speakerPin 3

#include <avr/pgmspace.h>
#include <stdint.h>

#include "wavetable.h"
#include "sustain.h"
#include "envelope.h"
#include "scales.h"
#include "tune.h"

#define SAMPLERATE 6250
#define ENVELOPEDURATION 1500 // miliseconds
#define OSCILLATOR_COUNT 6 // number of simultaneous notes, 256 max or 8 bit accumulation will overflow 16 bits
#define SIMULTANEOUS_NOTES 6 // generated samples are divided by this value, lots of clipping will happen if too low, volume will be low if too high
#define CLIP 127

#define BPM 190 // beats per minute
#define QUARTER 120 // note resolution in midi file

// some data use fixed point values, the 4 lower bits are used as the value after the decimal point
// round is achieved by dividing by 16 (or doing a right shift of 4 bits on unsigned values)

volatile char nboverflow=0;

// configure timer2 to generate 62500Hz pwm on digital pin 3 (OC2B)
void pwm_init()
{
  // disable TCNT2 interrupts
  TIMSK2 = 0;

  OCR2B = 128;
  
  // set Fast PWM mode, non inverting
  TCCR2A = 35; // COM2B1 + WGM21 + WGM20;
//    TCCR2A = 35+16; // COM2B1 + COM2B0 + WGM21 + WGM20; // inverting

  //Set the appropriate prescaler bits
  TCCR2B = 1; // no prescaling
  
  // enable TCNT2 overflow interrupt
  TIMSK2 |= (1<<TOIE2);
}

void setup() {
  pinMode (speakerPin, OUTPUT);
 }

// timer2 will overflow every 1/62500 s (16 microseconds)
ISR(TIMER2_OVF_vect) {
  nboverflow++;
}  

void loop() {
    pwm_init();
    
    uint16_t increments[ OSCILLATOR_COUNT ]; // 4 lower bits are fixed point value
    uint32_t phase_accu[ OSCILLATOR_COUNT ]; // same
    uint16_t envelope_positions[ OSCILLATOR_COUNT ]; // same
    uint8_t sustaining[ OSCILLATOR_COUNT ];  // true if the attack is finished and the channel is now using the sustain wavetable
    uint8_t playing [ OSCILLATOR_COUNT ];    // true if the channel is playing, playing stops when the sound has faded because of the envelope
	
    uint8_t next_osc = 0; // the next free (or not) channel
    uint16_t ticks = 0; // each tick we compute a sample
    uint16_t time = 120;  // tune time
    uint16_t event_index = 0; // index in the tune
    uint32_t totalticks =0;
	
    const uint16_t event_count = sizeof(tune_time);
    uint16_t sizeof_wt = sizeof( wt_attack );
    uint16_t sizeof_wtx16 = sizeof_wt << 4;
    const uint16_t sizeof_wt_sustain = sizeof( wt_sustain );
    uint16_t sizeof_wt_sustainx16 = sizeof_wt_sustain << 4;
    const uint16_t sizeof_envelope_table = sizeof( envelope_table );
	
    uint16_t envelopeinc = sizeof_envelope_table*256ul*1000ul /ENVELOPEDURATION/SAMPLERATE;
    envelopeinc=2;
    
    const uint16_t sustainfreq = SAMPLERATE / sizeof_wt_sustain;
    uint16_t TICKS_LIMIT = SAMPLERATE * 60ul / (BPM * QUARTER); // ticks in a time unit of the tune



    // initialization, set everything to 0
    uint8_t osc;
    for ( osc = 0; osc < OSCILLATOR_COUNT; ++osc ) {
        increments[ osc ] = 0;
        phase_accu[ osc ] = 0;
        envelope_positions[ osc ] = 0;
	sustaining[ osc ] = 0;
	playing [ osc ] = 0;
    }

    while ( 1 ) {
        while ( time >= pgm_read_word_near (tune_time+event_index) ) { // time reached the next note(s), let's assign an oscillator to it(them)
			// base sample frequency = SAMPLERATE / sizeof_wt_sustain
			// increment = note frequency / base frequency
            uint8_t pitch = pgm_read_byte_near(tune_pitch+event_index); // play one octave higher
            increments[ next_osc ] = (pgm_read_word_near(scale_table+pitch) << 4) / sustainfreq;
            phase_accu[ next_osc ] = 0;
            envelope_positions[ next_osc ] = 0;
	    sustaining [ next_osc ] = 0;
	    playing [ next_osc ] = 1;
            ++next_osc;
            if ( next_osc >= OSCILLATOR_COUNT ) {
                next_osc = 0;
            }
            ++event_index;
	  // if end of the song, restart
            if ( event_index >= event_count ) {
                ticks = 0;
                time = 0;
                event_index = 0;
            }
        }
        ++ticks;
        totalticks++;
        if ( ticks >= TICKS_LIMIT ) {
	  ticks=0;
            time += 1;
        }

        int16_t value = 0;
        for ( osc = 0; osc < OSCILLATOR_COUNT; ++osc ) {
			if (playing [ osc ] ) {
				
				phase_accu[ osc ] += increments[ osc ]; // advance phase
                                uint8_t envelope;
                                int8_t sustain;
				if (sustaining [ osc]) { // if sustaining
					while ( (phase_accu[ osc ] >> 4 )>= sizeof_wt_sustain ) { // check if beyond table
						phase_accu[ osc ] -= sizeof_wt_sustainx16;
					}
                                        uint16_t envpos = envelope_positions[osc] >>4;
					if ( (envpos) >= sizeof_envelope_table - 1 ) {
						// sound has decayed, no need to play it anymore.
						playing [ osc ] = 0;
					} else {
                                                envelope = pgm_read_byte_near (envelope_table + envpos);
                                                sustain = pgm_read_byte_near (wt_sustain + (phase_accu[osc]>>4));
                                                int16_t temp;
                                                if (sustain < 0) {
                                                  temp = -sustain;
                                                  temp *=envelope;
                                                  value -= temp >> 8;
                                                } else {
                                                  temp = sustain;
                                                  temp *= envelope;
						  value += temp >> 8; // apply envelope only if sustaining
                                                }
						envelope_positions [ osc ] += envelopeinc;
					}
				} else {
					if ( (phase_accu[ osc ] >>4) >= sizeof_wt ) { // check if beyond table
						sustaining [ osc ] = 1;
						phase_accu[ osc ] -= sizeof_wtx16;
						// check if beyond sustain table too
						while ( (phase_accu[ osc ] >> 4) >= sizeof_wt_sustain ) {
							phase_accu[ osc ] -= sizeof_wt_sustainx16;
						}
                                                sustain = pgm_read_byte_near (wt_sustain + (phase_accu[osc]>>4));
						value += sustain;
					} else {
                                                sustain = pgm_read_byte_near (wt_attack + (phase_accu[osc]>>4));
						value += sustain;
					}
				}			
			}
        }
        value /= SIMULTANEOUS_NOTES;
        if ( value > CLIP ) {
            value = CLIP;
        } else if ( value < -CLIP ) {
            value = -CLIP;
        }
        while (nboverflow <10);
        OCR2B = value+128;
        nboverflow=0;
    }
}

