// yamb: yet another music box
#include <stdio.h>

#include "wavetable.h"
#include "sustain.h"
#include "envelope.h"
#include "scales.h"
#include "tune.h"

#define SAMPLERATE 15625
#define ENVELOPEDURATION 1500 // miliseconds
#define OSCILLATOR_COUNT 6 // number of simultaneous notes, 256 max or 8 bit accumulation will overflow 16 bits
#define SIMULTANEOUS_NOTES 6 // generated samples are divided by this value, lots of clipping will happen if too low, volume will be low if too high
#define CLIP 127

#define BPM 190 // beats per minute
#define QUARTER 120 // note resolution in midi file

int main(int argc, char **argv)
{
    uint16_t increments[ OSCILLATOR_COUNT ]; // 4 lower bits are fixed point value
    uint32_t phase_accu[ OSCILLATOR_COUNT ]; // same
    uint16_t envelope_positions[ OSCILLATOR_COUNT ]; // same
	uint8_t sustaining[ OSCILLATOR_COUNT ];  // true if the attack is finished and the channel is now using the sustain wavetable
	uint8_t playing [ OSCILLATOR_COUNT ];    // true if the channel is playing, playing stops when the sound has faded because of the envelope
	
    uint8_t next_osc = 0; // the next free (or not) channel
    uint16_t ticks = 0; // each tick we compute a sample
    uint16_t time = 120;  // tune time
    uint16_t event_index = 0; // index in the tune
	uint32_t totalticks=0;
	
    const uint16_t event_count = sizeof(tune) / sizeof( tune[ 0 ] );
    const uint16_t sizeof_wt = sizeof( wt_attack );
    const uint16_t sizeof_wt_sustain = sizeof( wt_sustain );
    const uint16_t sizeof_envelope_table = sizeof( envelope_table );
	
	uint16_t envelopeinc = sizeof_envelope_table*256*1000 /(ENVELOPEDURATION*SAMPLERATE);
	envelopeinc = 2;
	const uint16_t sustainfreq = SAMPLERATE / sizeof_wt_sustain;
	const uint16_t TICKS_LIMIT = SAMPLERATE * 60 / (BPM * QUARTER); // ticks in a time unit of the tune
	
	// initialization, set everything to 0
	uint8_t osc;
    for ( osc = 0; osc < OSCILLATOR_COUNT; ++osc ) {
        increments[ osc ] = 0;
        phase_accu[ osc ] = 0;
        envelope_positions[ osc ] = 0;
		sustaining[ osc ] = 0;
		playing [ osc ] = 0;
    }
    while ( 1 ) {
        while ( time >= tune[ event_index ].time ) { // time reached the next note(s), let's assign an oscillator to it(them)
			// base sample frequency = SAMPLERATE / sizeof_wt_sustain
			// increment = note frequency / base frequency
            increments[ next_osc ] = scale_table[ tune[ event_index ].pitch +12 ] * 16 / sustainfreq;
//			printf (" Increment: %d\n", increments[next_osc]);
            phase_accu[ next_osc ] = 0;
            envelope_positions[ next_osc ] = 0;
			sustaining [ next_osc ] = 0;
			playing [ next_osc ] = 1;
            ++next_osc;
            if ( next_osc >= OSCILLATOR_COUNT ) {
                next_osc = 0;
            }
            ++event_index;
			// if end of the song, restart
            if ( event_index >= event_count ) {
                ticks = 0;
                time = 0;
                event_index = 0;
				// remove this to loop endlessly
				return 0;
            }
        }
        ++ticks;
		totalticks++;
        if ( ticks >= TICKS_LIMIT ) {
			ticks=0;
            time += 1;
        }

        int16_t value = 0;
        for ( osc = 0; osc < OSCILLATOR_COUNT; ++osc ) {
			if (playing [ osc ] ) {
				
				phase_accu[ osc ] += increments[ osc ]; // advance phase
//		printf (" phase accu %d ", (phase_accu[ osc ] ));
				if (sustaining [ osc]) { // if sustaining
					while ( (phase_accu[ osc ] / 16 )>= sizeof_wt_sustain ) { // check if beyond table
						phase_accu[ osc ] -= sizeof_wt_sustain * 16;
					}
					if ( (envelope_positions[ osc ] / 16) >= sizeof_envelope_table - 1 ) {
						// sound has decayed, no need to play it anymore.
						playing [ osc ] = 0;
					} else {
		//printf ("envelope %d ", envelope_table[ (envelope_positions[ osc ] / 16)]);
		//printf (" envelope position %d ", (envelope_positions[ osc ] ));
		//printf ("sustain %d\n", wt_sustain[ phase_accu[ osc ] / 16 ]);
						value += (envelope_table[ (envelope_positions[ osc ] / 16) ] * wt_sustain[ phase_accu[ osc ] / 16 ]) / 256; // apply envelope only if sustaining
						envelope_positions [ osc ] += envelopeinc;
					}
				} else {
					if ( (phase_accu[ osc ] / 16) >= sizeof_wt ) { // check if beyond table
						sustaining [ osc ] = 1;
						phase_accu[ osc ] -= sizeof_wt * 16;
						// check if beyond sustain table too
						while ( (phase_accu[ osc ] / 16) >= sizeof_wt_sustain ) {
							phase_accu[ osc ] -= sizeof_wt_sustain * 16;
						}
						value += wt_sustain[ phase_accu[ osc ] / 16 ];
					} else {
						value += wt_attack[ phase_accu[ osc ] / 16 ];
					}
				}			
			}
        }
		value /= SIMULTANEOUS_NOTES;
        if ( value > CLIP ) {
            value = CLIP;
        } else if ( value < -CLIP ) {
            value = -CLIP;
        }
		if (totalticks==2000)
			return 0;
		fputc (value, stdout);
//		printf ("% 4d", value);
    }

    return 0;
}
